<?php
/**
 * Public_Opinion_Lite functions and definitions
 */

if ( ! defined( 'PUBLIC_OPINION_LITE_NAME' ) ) {
	define( 'PUBLIC_OPINION_LITE_NAME', 'public-opinion-lite' );
}
if ( ! defined( 'PUBLIC_OPINION_LITE_WHITELABEL' ) ) {
	// Set the following to true, if you want to remove any user-facing CSSIgniter traces.
	define( 'PUBLIC_OPINION_LITE_WHITELABEL', false );
}

if ( ! function_exists( 'public_opinion_lite_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function public_opinion_lite_setup() {

	// Default content width.
	$GLOBALS['content_width'] = 885;

	// Remove filter-based base theme features not applicable in this theme.
	add_filter( 'public_opinion_lite_base_support_menu_2', '__return_false' );

	// Remove the base header layouts as they are not applicable to this theme. This effectively hides the customizer dropdown.
	add_filter( 'public_opinion_lite_header_layout_choices', '__return_empty_array' );

	// Add common theme supports.
	public_opinion_lite_base_setup();

	// Image sizes
	set_post_thumbnail_size( 885, 500, true );
	add_image_size( 'public_opinion_lite_item_hero_fullwidth', 1340, 500, true );
	add_image_size( 'public_opinion_lite_item_media', 885, 750, true );
	add_image_size( 'public_opinion_lite_fullwidth', 1140, 650, true );

	add_theme_support( 'public-opinion-lite-hide-single-featured', apply_filters( 'public_opinion_lite_theme_support_hide_single_featured_post_types', array(
		'post',
		'page',
	) ) );


	// Additional nav menus
	register_nav_menus( array(
		'menu-top' => esc_html__( 'Top menu', 'public-opinion-lite' ),
	) );

	// Remove the usual post meta and replace with item meta instead.
	remove_action( 'public_opinion_lite_the_post_header', 'public_opinion_lite_the_post_entry_meta', 20 );
	add_action( 'public_opinion_lite_the_post_header', 'public_opinion_lite_the_entry_item_top_meta', 5 );
	add_action( 'public_opinion_lite_the_post_header', 'public_opinion_lite_the_entry_item_author', 20 );

	add_action( 'public_opinion_lite_the_entry_item_top_meta', 'public_opinion_lite_the_entry_item_category', 10 );
	add_action( 'public_opinion_lite_the_entry_item_top_meta', 'public_opinion_lite_the_entry_item_time', 20 );
	add_action( 'public_opinion_lite_the_entry_item_top_meta', 'public_opinion_lite_the_entry_item_meta_info_open', 30 );
	add_action( 'public_opinion_lite_the_entry_item_top_meta', 'public_opinion_lite_the_entry_item_comments_number', 40 );
	add_action( 'public_opinion_lite_the_entry_item_top_meta', 'public_opinion_lite_the_entry_item_meta_info_close', 60 );

	// This provides back-compat for author descriptions on WP < 4.9. Remove by WP 5.1.
	if ( ! has_filter( 'get_the_author_description', 'wpautop' ) ) {
		add_filter( 'get_the_author_description', 'wpautop' );
	}
}
endif;
add_action( 'after_setup_theme', 'public_opinion_lite_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function public_opinion_lite_content_width() {
	$content_width = $GLOBALS['content_width'];

	if ( is_page_template( 'templates/full-width-page.php' )
		|| is_page_template( 'templates/front-page.php' )
		|| is_page_template( 'templates/builder.php' )
	) {
		$content_width = 1340;
	} elseif ( is_singular() || is_home() || is_archive() ) {
		$info          = public_opinion_lite_get_layout_info();
		$content_width = $info['content_width'];
	}

	$GLOBALS['content_width'] = apply_filters( 'public_opinion_lite_content_width', $content_width );
}
add_action( 'template_redirect', 'public_opinion_lite_content_width', 0 );

/**
 * Register widget areas.
 */
function public_opinion_lite_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Blog', 'public-opinion-lite' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Widgets added here will appear on the blog section.', 'public-opinion-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Page', 'public-opinion-lite' ),
		'id'            => 'sidebar-2',
		'description'   => esc_html__( 'Widgets added here will appear on the static pages.', 'public-opinion-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Front Page', 'public-opinion-lite' ),
		'id'            => 'frontpage',
		'description'   => esc_html__( 'These widgets appear on pages that have the "Front page" template assigned.', 'public-opinion-lite' ),
		'before_widget' => '<section id="%1$s" class="widget-section %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="section-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Front Page - Side', 'public-opinion-lite' ),
		'id'            => 'frontpage-side',
		'description'   => esc_html__( 'These widgets appear on the side of pages that have the "Front page" template assigned.', 'public-opinion-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Ad - Header', 'public-opinion-lite' ),
		'id'            => 'ad-header',
		'description'   => esc_html__( 'Advertisement area in the header. Use this area only for your banners.', 'public-opinion-lite' ),
		'before_widget' => '',
		'after_widget'  => '',
		'before_title'  => '<span class="screen-reader-text">',
		'after_title'   => '</span>',
	) );
}
add_action( 'widgets_init', 'public_opinion_lite_widgets_init' );


function public_opinion_lite_load_widgets() {
	require get_template_directory() . '/inc/widgets/home-instagram.php';
	require get_template_directory() . '/inc/widgets/home-featured-posts.php';
	require get_template_directory() . '/inc/widgets/home-category-featured-posts.php';
	require get_template_directory() . '/inc/widgets/featured-posts.php';

	if ( class_exists( 'CI_Widget_Home_Instagram' ) ) {
		register_widget( 'CI_Widget_Home_Instagram' );
	}

	register_widget( 'CI_Widget_Home_Category_Featured_Posts' );
	register_widget( 'CI_Widget_Home_Featured_Posts' );
	register_widget( 'CI_Widget_Featured_Posts' );
}
add_action( 'widgets_init', 'public_opinion_lite_load_widgets' );


function public_opinion_lite_get_fullwidth_sidebars() {
	return apply_filters( 'public_opinion_lite_fullwidth_sidebars', array(
		'frontpage',
	) );
}


function public_opinion_lite_get_fullwidth_widgets() {
	return apply_filters( 'public_opinion_lite_fullwidth_widgets', array(
		'ci-home-instagram',
		'ci-home-category-featured-posts',
		'ci-home-featured-posts',
	) );
}

add_filter( 'public_opinion_lite_has_sidebar', 'public_opinion_lite_has_sidebar_frontpage' );
function public_opinion_lite_has_sidebar_frontpage( $has_sidebar ) {
	if ( is_page_template( 'templates/front-page.php' ) ) {
		return is_active_sidebar( 'frontpage-side' ) ? true : false;
	}

	return $has_sidebar;
}


/**
 * Register Google Fonts
 */
function public_opinion_lite_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';

	/* translators: If there are characters in your language that are not supported by Lato, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Lato font: on or off', 'public-opinion-lite' ) ) {
		$fonts[] = 'Lato:400,400i,700';
	}

	/* translators: If there are characters in your language that are not supported by Roboto Condensed, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Roboto Condensed font: on or off', 'public-opinion-lite' ) ) {
		$fonts[] = 'Roboto Condensed:400,700';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
		), 'https://fonts.googleapis.com/css' );
	}

	return $fonts_url;
}

/**
 * Enqueue scripts and styles.
 */
function public_opinion_lite_scripts() {
	$theme = wp_get_theme();

	wp_register_style( 'public-opinion-lite-google-font', public_opinion_lite_fonts_url(), array(), null );
	wp_register_style( 'public-opinion-lite-base', get_template_directory_uri() . '/css/base.css', array(), $theme->get( 'Version' ) );
	wp_register_style( 'mmenu', get_template_directory_uri() . '/css/mmenu.css', array(), '5.5.3' );

	wp_register_style( 'public-opinion-lite-dependencies', false, array(
		'public-opinion-lite-google-font',
		'public-opinion-lite-base',
		'public-opinion-lite-common',
		'mmenu',
		'slick',
		'font-awesome',
	), $theme->get( 'Version' ) );

	if ( is_child_theme() ) {
		wp_enqueue_style( 'public-opinion-lite-style-parent', get_template_directory_uri() . '/style.css', array(
			'public-opinion-lite-dependencies',
		), $theme->get( 'Version' ) );
	}

	wp_enqueue_style( 'public-opinion-lite-style', get_stylesheet_uri(), array(
		'public-opinion-lite-dependencies',
	), $theme->get( 'Version' ) );


	wp_register_script( 'mmenu', get_template_directory_uri() . '/js/jquery.mmenu.min.all.js', array( 'jquery' ), '5.5.3', true );
	wp_register_script( 'fitVids', get_template_directory_uri() . '/js/jquery.fitvids.js', array( 'jquery' ), '1.1', true );
	wp_register_script( 'isotope', get_template_directory_uri() . '/js/isotope.pkgd.min.js', array( 'jquery' ), '3.0.2', true );

	/*
	 * Enqueue
	 */
	wp_enqueue_script( 'public-opinion-lite-front-scripts', get_template_directory_uri() . '/js/scripts.js', array(
		'jquery',
		'mmenu',
		'slick',
		'fitVids',
		'isotope',
		'anim-on-scroll',
	), $theme->get( 'Version' ), true );
}
add_action( 'wp_enqueue_scripts', 'public_opinion_lite_scripts' );


/**
 * Enqueue admin scripts and styles.
 */
function public_opinion_lite_admin_scripts( $hook ) {
	$theme = wp_get_theme();

	//
	// Styles
	//
	wp_register_style( 'public-opinion-lite-widgets', get_template_directory_uri() . '/css/admin/widgets.css', array(
		'public-opinion-lite-repeating-fields',
		'public-opinion-lite-plugin-post-meta',
		'alpha-color-picker',
	), $theme->get( 'Version' ) );


	//
	// Scripts
	//
	wp_register_script( 'public-opinion-lite-widgets', get_template_directory_uri() . '/js/admin/widgets.js', array(
		'jquery',
		'public-opinion-lite-repeating-fields',
		'public-opinion-lite-plugin-post-meta',
		'alpha-color-picker',
	), $theme->get( 'Version' ), true );
	$params = array(
		'ajaxurl' => admin_url( 'admin-ajax.php' ),
	);
	wp_localize_script( 'public-opinion-lite-widgets', 'ThemeWidget', $params );


	/*
	 * Enqueue
	 */
	if ( in_array( $hook, array( 'widgets.php', 'customize.php' ), true ) ) {
		wp_enqueue_media();
		wp_enqueue_style( 'public-opinion-lite-widgets' );
		wp_enqueue_script( 'public-opinion-lite-widgets' );
	}

}
add_action( 'admin_enqueue_scripts', 'public_opinion_lite_admin_scripts' );

add_action( 'elementor/editor/before_enqueue_scripts', 'public_opinion_lite_enqueue_scripts_elementor' );
function public_opinion_lite_enqueue_scripts_elementor() {
	public_opinion_lite_base_register_scripts();
	public_opinion_lite_admin_scripts( '' );
	wp_enqueue_media();
	wp_enqueue_style( 'public-opinion-lite-widgets' );
	wp_enqueue_script( 'public-opinion-lite-widgets' );
}

add_filter( 'wp_page_menu', 'public_opinion_lite_wp_page_menu', 10, 2 );
function public_opinion_lite_wp_page_menu( $menu, $args ) {
	$menu = preg_replace( '#^<div .*?>#', '', $menu, 1 );
	$menu = preg_replace( '#</div>$#', '', $menu, 1 );
	$menu = preg_replace( '#^<ul>#', '<ul id="' . esc_attr( $args['menu_id'] ) . '" class="' . esc_attr( $args['menu_class'] ) . '">', $menu, 1 );
	return $menu;
}

if ( ! function_exists( 'public_opinion_lite_get_columns_classes' ) ) :
	function public_opinion_lite_get_columns_classes( $columns ) {
		switch ( intval( $columns ) ) {
			case 1:
				$classes = 'col-12';
				break;
			case 2:
				$classes = 'col-sm-6 col-12';
				break;
			case 3:
				$classes = 'col-lg-4 col-sm-6 col-12';
				break;
			case 4:
			default:
				$classes = 'col-xl-3 col-lg-4 col-sm-6 col-12';
				break;
		}

		return apply_filters( 'public_opinion_lite_get_columns_classes', $classes, $columns );
	}
endif;


function public_opinion_lite_get_hero_articles_layouts() {
	$layouts = array(
		'two-posts' => array(
			'title' => __( 'Two posts', 'public-opinion-lite' ),
			'posts' => 2,
		),
		'three-posts' => array(
			'title' => __( 'Three posts', 'public-opinion-lite' ),
			'posts' => 3,
		),
	);

	return apply_filters( 'public_opinion_lite_hero_articles_layouts', $layouts );
}

function public_opinion_lite_get_hero_articles_layout_choices() {
	$choices = array();

	foreach ( public_opinion_lite_get_hero_articles_layouts() as $key => $choice ) {
		$choices[ $key ] = $choice['title'];
	}

	return apply_filters( 'public_opinion_lite_hero_articles_layout_choices', $choices );
}

function public_opinion_lite_sanitize_hero_articles_layout( $value ) {
	$choices = public_opinion_lite_get_hero_articles_layout_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return '';
}

function public_opinion_lite_get_hero_articles_ids( $layout, $category_id = false, $query_args = array() ) {
	$layouts = public_opinion_lite_get_hero_articles_layouts();
	if ( ! array_key_exists( $layout, $layouts ) ) {
		return array();
	}

	$featured_tag = get_theme_mod( 'theme_hero_tag', 'featured' );
	if ( empty( $featured_tag ) ) {
		return array();
	}

	$posts_per_page = $layouts[ $layout ]['posts'];

	$q_args = array(
		'posts_per_page'      => $posts_per_page,
		'tag'                 => $featured_tag,
		'ignore_sticky_posts' => true,
		'fields'              => 'ids',
	);

	if ( intval( $category_id ) > 0 ) {
		$q_args['cat'] = intval( $category_id );
	}

	$q_args = apply_filters( 'public_opinion_lite_get_hero_articles_ids_args', array_merge( $q_args, $query_args ), $q_args, $query_args );

	$q = new WP_Query( $q_args );

	return $q->posts;
}

function public_opinion_lite_get_hero_articles( $layout, $category_id = false, $query_args = array() ) {
	$ids = public_opinion_lite_get_hero_articles_ids( $layout, $category_id, $query_args );

	if ( empty( $ids ) ) {
		return false;
	}

	return new WP_Query( array(
		'posts_per_page'      => - 1,
		'post__in'            => $ids,
		'ignore_sticky_posts' => true,
		'orderby'             => 'post__in',
	) );
}

function public_opinion_lite_get_category_articles_layouts() {
	/* translators: %d is a layout number, e.g. Layout #1 */
	$title = __( 'Layout #%d', 'public-opinion-lite' );

	/* translators: %1$d is a layout number, e.g. Layout #1. %2$d is the exact number of posts required.  */
	$title_max = __( 'Layout #%1$d (%2$d posts)', 'public-opinion-lite' );

	$layouts = array(
		2 => array(
			'posts' => -1,
		),
	);

	foreach ( $layouts as $key => $value ) {
		if ( - 1 === $value['posts'] ) {
			$layouts[ $key ]['title'] = esc_html( sprintf( $title, $key ) );
		} else {
			$layouts[ $key ]['title'] = esc_html( sprintf( $title_max, $key, $value['posts'] ) );
		}
	}

	return apply_filters( 'public_opinion_lite_category_articles_layouts', $layouts );
}


function public_opinion_lite_get_category_articles_layout_choices() {
	$choices = array();

	foreach ( public_opinion_lite_get_category_articles_layouts() as $key => $choice ) {
		$choices[ $key ] = $choice['title'];
	}

	return apply_filters( 'public_opinion_lite_category_articles_layout_choices', $choices );
}

function public_opinion_lite_sanitize_category_articles_layout( $value ) {
	$choices = public_opinion_lite_get_category_articles_layout_choices();
	if ( array_key_exists( intval( $value ), $choices ) ) {
		return $value;
	}

	return '';
}

function public_opinion_lite_get_category_layout_articles_ids( $layout, $category_id, $type = 'featured', $posts_per_page = false, $query_args = array() ) {
	$layouts = public_opinion_lite_get_category_articles_layouts();
	if ( ! array_key_exists( $layout, $layouts ) ) {
		return array();
	}

	$q_args = array(
		'ignore_sticky_posts' => true,
		'fields'              => 'ids',
	);

	if ( intval( $category_id ) > 0 ) {
		$q_args['cat'] = intval( $category_id );
	}

	if ( 'featured' === $type ) {
		$featured_tag = get_theme_mod( 'theme_hero_tag', 'featured' );
		if ( empty( $featured_tag ) ) {
			return array();
		}

		$q_args['tag'] = $featured_tag;
	}


	// Only use $posts_per_page if layout doesn't have a strict posts requirement.
	if ( -1 === $layouts[ $layout ]['posts'] && ! empty( $posts_per_page ) ) {
		$posts_per_page = intval( $posts_per_page );
	} elseif ( $layouts[ $layout ]['posts'] > 0 ) {
		$posts_per_page = $layouts[ $layout ]['posts'];
	}

	if ( ! empty( $posts_per_page ) && $posts_per_page > 0 ) {
		$q_args['posts_per_page'] = intval( $posts_per_page );
	}


	$q_args = apply_filters( 'public_opinion_lite_get_category_articles_ids_args', array_merge( $q_args, $query_args ), $q_args, $query_args );

	$q = new WP_Query( $q_args );

	return $q->posts;
}

function public_opinion_lite_get_category_layout_articles( $layout, $category_id = false, $type = 'featured', $posts_per_page = false, $query_args = array() ) {
	$ids = public_opinion_lite_get_category_layout_articles_ids( $layout, $category_id, $type, $posts_per_page, $query_args );

	if ( empty( $ids ) ) {
		return false;
	}

	return new WP_Query( array(
		'posts_per_page'      => - 1,
		'post__in'            => $ids,
		'ignore_sticky_posts' => true,
		'orderby'             => 'post__in',
	) );
}



add_filter( 'public_opinion_lite_get_primary_post_category', 'public_opinion_lite_get_primary_post_category_yoast_seo', 10, 3 );
function public_opinion_lite_get_primary_post_category_yoast_seo( $term, $post_id, $terms ) {
	if ( empty( $terms ) ) {
		return $term;
	}

	$term_ids = wp_list_pluck( $terms, 'term_id' );

	$primary_id = intval( get_post_meta( $post_id, '_yoast_wpseo_primary_category', true ) );

	if ( $primary_id > 0 && in_array( $primary_id, $term_ids, true ) ) {
		$term = get_term( $primary_id );
	}

	return $term;
}

function public_opinion_lite_get_the_post_category_color( $post = null ) {
	// Category colors are only supported on Public Opinion (premium). Please upgrade.
	return false;
}

// This should run after public_opinion_lite_get_all_customizer_css(), i.e. add_action( 'wp_enqueue_scripts', 'public_opinion_lite_base_enqueue_scripts' );
add_action( 'wp_enqueue_scripts', 'public_opinion_lite_enqueue_post_category_colors', 20 );
function public_opinion_lite_enqueue_post_category_colors() {
	if ( ! is_singular( 'post' ) ) {
		return;
	}

	$color = public_opinion_lite_get_the_post_category_color();

	if ( empty( $color ) ) {
		return;
	}

	$selector = apply_filters( 'public_opinion_lite_the_post_category_color_css_selector', '.entry-content a, .entry-content blockquote, .entry-fields, .entry-rating-scores' );
	$rules    = apply_filters( 'public_opinion_lite_the_post_category_color_css_rules', '%1$s { color: %2$s; }', $color, $selector );
	$css = sprintf( $rules,
		$selector,
		sanitize_hex_color( $color )
	);

	wp_add_inline_style( 'public-opinion-lite-style', $css );
}

add_filter( 'wp_link_pages_args', 'public_opinion_lite_wp_link_pages_args' );
function public_opinion_lite_wp_link_pages_args( $params ) {
	$params = array_merge( $params, array(
		'before'         => '<nav class="navigation post-navigation" role="navigation"><h2 class="screen-reader-text">' . esc_html__( 'Multi-page post navigation:', 'public-opinion-lite' ) . '</h2><div class="nav-links">',
		'after'          => '</div></nav>',
		'next_or_number' => 'next',
	) );

	return $params;
}

add_filter( 'wp_link_pages_link', 'public_opinion_lite_wp_link_pages_link', 10, 2 );
function public_opinion_lite_wp_link_pages_link( $link, $next ) {
	global $page;
	if ( $page < $next ) {
		preg_match( '#<a (.*?)>(.*)</a>#', $link, $matches );
		$link = '<div class="nav-next"><a ' . $matches[1] . '>' . $matches[2] . '</a></div>';
	} elseif ( $page > $next ) {
		preg_match( '#<a (.*?)>(.*)</a>#', $link, $matches );
		$link = '<div class="nav-previous"><a ' . $matches[1] . '>' . $matches[2] . '</a></div>';
	}

	return $link;
}


/**
 * Ajax handler for tag search.
 *
 * This is a copy of wp_ajax_ajax_tag_search() but modified to return the tag slugs instead.
 *
 * @since 3.1.0
 */
function public_opinion_lite_ajax_tag_search() {
	if ( ! isset( $_GET['tax'] ) ) {
		wp_die( 0 );
	}

	$taxonomy = sanitize_key( $_GET['tax'] );
	$tax = get_taxonomy( $taxonomy );
	if ( ! $tax ) {
		wp_die( 0 );
	}

	if ( ! current_user_can( $tax->cap->assign_terms ) ) {
		wp_die( -1 );
	}

	$s = wp_unslash( $_GET['q'] );

	$comma = _x( ',', 'tag delimiter', 'public-opinion-lite' );
	if ( ',' !== $comma )
		$s = str_replace( $comma, ',', $s );
	if ( false !== strpos( $s, ',' ) ) {
		$s = explode( ',', $s );
		$s = $s[count( $s ) - 1];
	}
	$s = trim( $s );

	/**
	 * Filters the minimum number of characters required to fire a tag search via Ajax.
	 *
	 * @since 4.0.0
	 *
	 * @param int         $characters The minimum number of characters required. Default 2.
	 * @param WP_Taxonomy $tax        The taxonomy object.
	 * @param string      $s          The search term.
	 */
	$term_search_min_chars = (int) apply_filters( 'term_search_min_chars', 2, $tax, $s );

	/*
	 * Require $term_search_min_chars chars for matching (default: 2)
	 * ensure it's a non-negative, non-zero integer.
	 */
	if ( ( $term_search_min_chars == 0 ) || ( strlen( $s ) < $term_search_min_chars ) ){
		wp_die();
	}

	$results = get_terms( $taxonomy, array( 'search' => $s, 'fields' => 'id=>slug', 'hide_empty' => false ) );

	echo join( $results, "\n" );
	wp_die();
}
add_action( 'wp_ajax_public_opinion_lite-ajax-tag-search', 'public_opinion_lite_ajax_tag_search' );


// Hide the featured tag.
add_filter( 'get_terms', 'public_opinion_lite_get_terms_hide_featured_tag', 10, 3 );
add_filter( 'get_the_terms', 'public_opinion_lite_get_the_terms_hide_featured_tag', 10, 3 );

function public_opinion_lite_get_terms_hide_featured_tag( $terms, $taxonomies, $args ) {
	if ( is_admin() ) {
		return $terms;
	}

	if ( ! in_array( 'post_tag', $taxonomies, true ) ) {
		return $terms;
	}

	if ( empty( $terms ) ) {
		return $terms;
	}

	if ( 'all' !== $args['fields'] ) {
		return $terms;
	}

	$featured_tag = get_theme_mod( 'theme_hero_tag', 'featured' );

	if ( $featured_tag ) {
		foreach ( $terms as $key => $term ) {
			if ( is_object( $term ) && $featured_tag === $term->slug ) {
				unset( $terms[ $key ] );
			}
		}
	}

	return $terms;
}

function public_opinion_lite_get_the_terms_hide_featured_tag( $terms, $id, $taxonomy ) {
	if ( is_admin() ) {
		return $terms;
	}

	if ( 'post_tag' !== $taxonomy ) {
		return $terms;
	}

	if ( empty( $terms ) ) {
		return $terms;
	}

	$featured_tag = get_theme_mod( 'theme_hero_tag', 'featured' );

	if ( $featured_tag ) {
		foreach ( $terms as $key => $term ) {
			if ( $featured_tag === $term->slug ) {
				unset( $terms[ $key ] );
			}
		}
	}

	return $terms;
}


add_action( 'wp_loaded', 'public_opinion_lite_remove_pages_from_search' );
function public_opinion_lite_remove_pages_from_search() {
	global $wp_post_types;
	if ( get_theme_mod( 'theme_archive_exclude_pages_from_search', 1 ) && isset( $wp_post_types['page'] ) ) {
		$wp_post_types['page']->exclude_from_search = true;
	}
}


if ( ! defined( 'PUBLIC_OPINION_LITE_WHITELABEL' ) || false === (bool) PUBLIC_OPINION_LITE_WHITELABEL ) {
	add_filter( 'pt-ocdi/import_files', 'public_opinion_lite_ocdi_import_files' );
	add_action( 'pt-ocdi/after_import', 'public_opinion_lite_ocdi_after_import_setup' );
}

add_filter( 'pt-ocdi/timeout_for_downloading_import_file', 'public_opinion_lite_ocdi_download_timeout' );
function public_opinion_lite_ocdi_download_timeout( $timeout ) {
	return 60;
}

function public_opinion_lite_ocdi_import_files( $files ) {
	if ( ! defined( 'PUBLIC_OPINION_LITE_NAME' ) ) {
		return $files;
	}

	$demo_dir_url = untrailingslashit( apply_filters( 'public_opinion_lite_ocdi_demo_dir_url', 'https://www.cssigniter.com/sample_content/' . PUBLIC_OPINION_LITE_NAME ) );

	// When having more that one predefined imports, set a preview image, preview URL, and categories for isotope-style filtering.
	$new_files = array(
		array(
			'import_file_name'           => esc_html__( 'Demo Import', 'public-opinion-lite' ),
			'import_file_url'            => $demo_dir_url . '/content.xml',
			'import_widget_file_url'     => $demo_dir_url . '/widgets.wie',
			'import_customizer_file_url' => $demo_dir_url . '/customizer.dat',
		),
	);

	return array_merge( $files, $new_files );
}

function public_opinion_lite_ocdi_after_import_setup() {
	// Set up nav menus.
	$main_menu  = get_term_by( 'name', 'Main', 'nav_menu' );
	$top_menu   = get_term_by( 'name', 'Top', 'nav_menu' );

	set_theme_mod( 'nav_menu_locations', array(
		'menu-1'   => $main_menu->term_id,
		'menu-top' => $top_menu->term_id,
	) );

	// Set up home and blog pages.
	$front_page_id = get_page_by_title( 'News' );
	$blog_page_id  = get_page_by_title( 'Blog' );

	update_option( 'show_on_front', 'page' );
	update_option( 'page_on_front', $front_page_id->ID );
	update_option( 'page_for_posts', $blog_page_id->ID );
}

add_action( 'tgmpa_register', 'public_opinion_lite_register_required_plugins' );
function public_opinion_lite_register_required_plugins() {
	$plugins = array(
		array(
			'name'               => 'Public Opinion Lite theme-specific plugin', // The plugin name.
			'version'            => '1.0', // E.g. 1.0.0. If set, the active plugin must be this version or higher. If the plugin version is higher than the plugin version installed, the user will be notified to update the plugin.
			'slug'               => 'public-opinion-lite-plugin', // The plugin slug (typically the folder name).
			'source'             => get_template_directory() . '/plugins/public-opinion-lite-plugin.zip', // The plugin source.
			'required'           => true, // If false, the plugin is only 'recommended' instead of required.
			'force_activation'   => false, // If true, plugin is activated upon theme activation and cannot be deactivated until theme switch.
			'force_deactivation' => false, // If true, plugin is deactivated upon theme switch, useful for theme-specific plugins.
			'external_url'       => '', // If set, overrides default API URL and points to an external URL.
			'is_callable'        => '', // If set, this callable will be be checked for availability to determine if a plugin is active.
		),
	);

	$config = array(
		'id'           => 'public-opinion-lite',              // Unique ID for hashing notices for multiple instances of TGMPA.
		'default_path' => '',                      // Default absolute path to bundled plugins.
		'menu'         => 'tgmpa-install-plugins', // Menu slug.
		'has_notices'  => true,                    // Show admin notices or not.
		'dismissable'  => true,                    // If false, a user cannot dismiss the nag message.
		'dismiss_msg'  => '',                      // If 'dismissable' is false, this message will be output at top of nag.
		'is_automatic' => false,                   // Automatically activate plugins after installation or not.
		'message'      => '',                      // Message to output right before the plugins table.
	);

	tgmpa( $plugins, $config );
}

/**
 * Common theme features.
 */
require_once get_theme_file_path( '/common/common.php' );

/**
 * Base theme functions and definitions.
 */
require_once get_theme_file_path( '/base/functions.php' );

/**
 * Customizer additions.
 */
require_once get_template_directory() . '/inc/customizer.php';

/**
 * Customizer styles.
 */
require_once get_theme_file_path( '/inc/customizer-styles.php' );

/**
 * TGM_Plugin_Activation class.
 */
require_once get_template_directory() . '/base/class-tgm-plugin-activation.php';

/**
 * Theme-specific template tags.
 */
require_once get_template_directory() . '/inc/template-tags.php';

/**
 * Public Opinion Elements
 */
if ( defined( 'ELEMENTOR_VERSION' ) && version_compare( PHP_VERSION, '5.4', '>=' ) ) {
	require_once( 'inc/elements.php' );
}
