<?php
/**
 * CSSIgniter theme base functions and definitions
 */

if ( ! function_exists( 'public_opinion_lite_base_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function public_opinion_lite_base_setup() {
	// Make theme available for translation.
	load_theme_textdomain( 'public-opinion-lite', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	// Let WordPress manage the document title.
	add_theme_support( 'title-tag' );

	// Enable support for Post Thumbnails on posts and pages.
	add_theme_support( 'post-thumbnails' );

	$menus = array(
		'menu-1' => esc_html__( 'Main Menu', 'public-opinion-lite' ),
		'menu-2' => esc_html__( 'Main Menu - Right', 'public-opinion-lite' ),
	);
	if ( ! apply_filters( 'public_opinion_lite_base_support_menu_2', true ) ) {
		unset( $menus['menu-2'] );
	}
	register_nav_menus( $menus );

	// Switch default core markup for search form, comment form, and comments to output valid HTML5.
	add_theme_support( 'html5', apply_filters( 'public_opinion_lite_add_theme_support_html5', array(
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) ) );

	// Add theme support for custom logos.
	add_theme_support( 'custom-logo', apply_filters( 'public_opinion_lite_add_theme_support_custom_logo', array() ) );

	// Set up the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'public_opinion_lite_custom_background_args', array() ) );

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );
}
endif;

/**
 * Register scripts and styles unconditionally.
 */
function public_opinion_lite_base_register_scripts() {
	$theme = wp_get_theme();

	if ( ! wp_script_is( 'alpha-color-picker', 'enqueued' ) && ! wp_script_is( 'alpha-color-picker', 'registered' ) ) {
		wp_register_style( 'alpha-color-picker', get_template_directory_uri() . '/base/assets/vendor/alpha-color-picker/alpha-color-picker.css', array(
			'wp-color-picker',
		), '1.0.0' );
		wp_register_script( 'alpha-color-picker', get_template_directory_uri() . '/base/assets/vendor/alpha-color-picker/alpha-color-picker.js', array(
			'jquery',
			'wp-color-picker',
		), '1.0.0', true );
	}

	if ( ! wp_script_is( 'slick', 'enqueued' ) && ! wp_script_is( 'slick', 'registered' ) ) {
		wp_register_style( 'slick', get_template_directory_uri() . '/base/assets/vendor/slick/slick.css', array(), '1.6.0' );
		wp_register_script( 'slick', get_template_directory_uri() . '/base/assets/vendor/slick/slick.js', array(
			'jquery',
		), '1.6.0', true );
	}

	if ( ! wp_script_is( 'public-opinion-lite-plugin-post-meta', 'enqueued' ) && ! wp_script_is( 'public-opinion-lite-plugin-post-meta', 'registered' ) ) {
		wp_register_style( 'public-opinion-lite-plugin-post-meta', get_template_directory_uri() . '/base/assets/css/post-meta.css', array(
			'alpha-color-picker',
		), $theme->get( 'Version' ) );
		wp_register_script( 'public-opinion-lite-plugin-post-meta', get_template_directory_uri() . '/base/assets/js/post-meta.js', array(
			'media-editor',
			'jquery',
			'jquery-ui-sortable',
			'alpha-color-picker',
		), $theme->get( 'Version' ), true );

		$settings = array(
			'ajaxurl'             => admin_url( 'admin-ajax.php' ),
			'tSelectFile'         => esc_html__( 'Select file', 'public-opinion-lite' ),
			'tSelectFiles'        => esc_html__( 'Select files', 'public-opinion-lite' ),
			'tUseThisFile'        => esc_html__( 'Use this file', 'public-opinion-lite' ),
			'tUseTheseFiles'      => esc_html__( 'Use these files', 'public-opinion-lite' ),
			'tUpdateGallery'      => esc_html__( 'Update gallery', 'public-opinion-lite' ),
			'tLoading'            => esc_html__( 'Loading...', 'public-opinion-lite' ),
			'tPreviewUnavailable' => esc_html__( 'Gallery preview not available.', 'public-opinion-lite' ),
			'tRemoveImage'        => esc_html__( 'Remove image', 'public-opinion-lite' ),
			'tRemoveFromGallery'  => esc_html__( 'Remove from gallery', 'public-opinion-lite' ),
		);
		wp_localize_script( 'public-opinion-lite-plugin-post-meta', 'public_opinion_lite_plugin_PostMeta', $settings );
	}

	wp_register_style( 'public-opinion-lite-repeating-fields', get_template_directory_uri() . '/base/assets/css/repeating-fields.css', array(), $theme->get( 'Version' ) );
	wp_register_script( 'public-opinion-lite-repeating-fields', get_template_directory_uri() . '/base/assets/js/repeating-fields.js', array(
		'jquery',
		'jquery-ui-sortable',
	), $theme->get( 'Version' ), true );

	wp_register_style( 'font-awesome', get_template_directory_uri() . '/base/assets/vendor/fontawesome/css/font-awesome.css', array(), '4.7.0' );

	wp_register_script( 'imagesLoaded', get_template_directory_uri() . '/base/assets/js/imagesloaded.pkgd.min.js', array( 'jquery' ), '4.1.3', true );
	wp_register_script( 'anim-on-scroll', get_template_directory_uri() . '/base/assets/js/anim-on-scroll.js', array(
		'jquery',
		'imagesLoaded',
	), '1.0.1', true );

	wp_register_script( 'public-opinion-lite-base-front-scripts', get_template_directory_uri() . '/base/assets/js/scripts.js', array(
		'jquery',
	), $theme->get( 'Version' ), true );

	wp_register_style( 'jquery-magnific-popup', get_template_directory_uri() . '/base/assets/vendor/magnific-popup/magnific.css', array(), '1.0.0' );
	wp_register_script( 'jquery-magnific-popup', get_template_directory_uri() . '/base/assets/vendor/magnific-popup/jquery.magnific-popup.js', array( 'jquery' ), '1.0.0', true );
	wp_register_script( 'public-opinion-lite-magnific-init', get_template_directory_uri() . '/base/assets/js/magnific-init.js', array( 'jquery' ), '20170530', true );

	wp_register_style( 'public-opinion-lite-base-style', '', apply_filters( 'public_opinion_lite_base_style_depends', array( 'public-opinion-lite-style' ) ), $theme->get( 'Version' ) );

}
add_action( 'init', 'public_opinion_lite_base_register_scripts' );


/**
 * Enqueue scripts and styles.
 */
function public_opinion_lite_base_enqueue_scripts() {

	wp_enqueue_style( 'public-opinion-lite-base-style' );
	wp_add_inline_style( 'public-opinion-lite-base-style', public_opinion_lite_get_all_customizer_css() );

	wp_enqueue_script( 'public-opinion-lite-base-front-scripts' );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	if ( get_theme_mod( 'theme_lightbox', 1 ) ) {
		wp_enqueue_style( 'jquery-magnific-popup' );
		wp_enqueue_script( 'jquery-magnific-popup' );
		wp_enqueue_script( 'public-opinion-lite-magnific-init' );
	}

}
add_action( 'wp_enqueue_scripts', 'public_opinion_lite_base_enqueue_scripts' );


function public_opinion_lite_base_admin_enqueue_scripts( $hook ) {
	$theme = wp_get_theme();

	if ( in_array( $hook, array( 'widgets.php', 'customize.php' ) ) ) {
		wp_enqueue_style( 'public-opinion-lite-repeating-fields' );
		wp_enqueue_script( 'public-opinion-lite-repeating-fields' );
	}
}
add_action( 'admin_enqueue_scripts', 'public_opinion_lite_base_admin_enqueue_scripts' );


function public_opinion_lite_base_register_sidebars() {
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 1st column', 'public-opinion-lite' ),
		'id'            => 'footer-1',
		'description'   => esc_html__( 'Widgets added here will appear on the first footer column.', 'public-opinion-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 2nd column', 'public-opinion-lite' ),
		'id'            => 'footer-2',
		'description'   => esc_html__( 'Widgets added here will appear on the second footer column.', 'public-opinion-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 3rd column', 'public-opinion-lite' ),
		'id'            => 'footer-3',
		'description'   => esc_html__( 'Widgets added here will appear on the third footer column.', 'public-opinion-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer - 4th column', 'public-opinion-lite' ),
		'id'            => 'footer-4',
		'description'   => esc_html__( 'Widgets added here will appear on the fourth footer column.', 'public-opinion-lite' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
}
add_action( 'widgets_init', 'public_opinion_lite_base_register_sidebars' );

function public_opinion_lite_base_load_widgets() {
	require get_template_directory() . '/base/widgets/socials.php';
	require get_template_directory() . '/base/widgets/contact.php';
	require get_template_directory() . '/base/widgets/schedule.php';
	require get_template_directory() . '/base/widgets/latest-post-type.php';

	register_widget( 'CI_Widget_Socials' );
	register_widget( 'CI_Widget_Contact' );
	register_widget( 'CI_Widget_Schedule' );
	register_widget( 'CI_Widget_Latest_Post_Type' );
}
add_action( 'widgets_init', 'public_opinion_lite_base_load_widgets' );


if ( ! function_exists( 'public_opinion_lite_get_all_customizer_css' ) ) :
	function public_opinion_lite_get_all_customizer_css() {
		$styles = array(
			'base-customizer' => public_opinion_lite_base_get_customizer_css(),
			'customizer'      => public_opinion_lite_get_customizer_css(),
			'base-hero'       => public_opinion_lite_base_get_hero_styles(),
		);

		$styles = apply_filters( 'public_opinion_lite_all_customizer_css', $styles );

		return implode( PHP_EOL, $styles );
	}
endif;

if ( ! function_exists( 'public_opinion_lite_has_sidebar' ) ) :
/**
 * Determine if a sidebar is being displayed.
 */
function public_opinion_lite_has_sidebar() {
	$has_sidebar = false;

	if ( is_home() || is_archive() ) {
		if ( get_theme_mod( 'archive_sidebar', 1 ) && is_active_sidebar( 'sidebar-1' ) ) {
			$has_sidebar = true;
		}
	} elseif ( ! is_page() && is_active_sidebar( 'sidebar-1' ) ) {
		$has_sidebar = true;
	} elseif ( is_page() && is_active_sidebar( 'sidebar-2' ) ) {
		$has_sidebar = true;
	}

	return apply_filters( 'public_opinion_lite_has_sidebar', $has_sidebar );
}
endif;

if ( ! function_exists( 'public_opinion_lite_get_layout_info' ) ) :
/**
 * Return appropriate layout information.
 */
function public_opinion_lite_get_layout_info() {
	$has_sidebar = public_opinion_lite_has_sidebar();

	$classes = array(
		'container_classes' => $has_sidebar ? 'col-lg-8 col-12' : 'col-xl-8 offset-xl-2 col-lg-10 offset-lg-1 col-12',
		'sidebar_classes'   => $has_sidebar ? 'col-xl-3 offset-xl-1 col-lg-4 col-12' : '',
		'content_width'     => 750,
		'has_sidebar'       => $has_sidebar,
	);

	if ( is_singular() ) {
		if ( 'left' === get_post_meta( get_the_ID(), 'public_opinion_lite_sidebar', true ) ) {
			$classes = array(
				'container_classes' => $has_sidebar ? 'col-lg-8 push-lg-4 col-12' : 'col-xl-8 offset-xl-2 col-lg-10 offset-lg-1 col-12',
				'sidebar_classes'   => $has_sidebar ? 'col-xl-3 offset-xl-1 pull-xl-9 col-lg-4 pull-lg-8 col-12' : '',
				'content_width'     => 750,
				'has_sidebar'       => $has_sidebar,
			);
		} elseif ( 'none' === get_post_meta( get_the_ID(), 'public_opinion_lite_sidebar', true ) ) {
			$classes = array(
				'container_classes' => 'col-xl-8 offset-xl-2 col-lg-10 offset-lg-1 col-12',
				'sidebar_classes'   => '',
				'content_width'     => 750,
				'has_sidebar'       => false,
			);
		}
	} elseif ( is_home() || is_archive() ) {
		// 1 will get default narrow fullwidth classes. 2 and 3 will get fullwidth.
		if ( 1 !== (int) get_theme_mod( 'archive_layout', public_opinion_lite_archive_layout_default() ) ) {
			if ( ! $has_sidebar ) {
				$classes = array(
					'container_classes' => 'col-12',
					'sidebar_classes'   => '',
					'content_width'     => 1140,
					'has_sidebar'       => false,
				);
			}
		}
	}

	return apply_filters( 'public_opinion_lite_layout_info', $classes, $has_sidebar );
}
endif;

/**
 * Echoes container classes based on whether
 * the current template has a visible sidebar or not
 */
function public_opinion_lite_the_container_classes() {
	$info = public_opinion_lite_get_layout_info();
	echo esc_attr( $info['container_classes'] );
}

/**
 * Echoes container classes based on whether
 * the current template has a visible sidebar or not
 */
function public_opinion_lite_the_sidebar_classes() {
	$info = public_opinion_lite_get_layout_info();
	echo esc_attr( $info['sidebar_classes'] );
}

function public_opinion_lite_get_social_networks() {
	return apply_filters( 'public_opinion_lite_social_networks', array(
		array(
			'name'  => 'facebook',
			'label' => esc_html__( 'Facebook', 'public-opinion-lite' ),
			'icon'  => 'fa-facebook',
		),
		array(
			'name'  => 'twitter',
			'label' => esc_html__( 'Twitter', 'public-opinion-lite' ),
			'icon'  => 'fa-twitter',
		),
		array(
			'name'  => 'instagram',
			'label' => esc_html__( 'Instagram', 'public-opinion-lite' ),
			'icon'  => 'fa-instagram',
		),
		array(
			'name'  => 'snapchat',
			'label' => esc_html__( 'Snapchat', 'public-opinion-lite' ),
			'icon'  => 'fa-snapchat',
		),
		array(
			'name'  => 'bloglovin',
			'label' => esc_html__( 'Bloglovin', 'public-opinion-lite' ),
			'icon'  => 'fa-heart',
		),
		array(
			'name'  => 'pinterest',
			'label' => esc_html__( 'Pinterest', 'public-opinion-lite' ),
			'icon'  => 'fa-pinterest',
		),
		array(
			'name'  => 'youtube',
			'label' => esc_html__( 'YouTube', 'public-opinion-lite' ),
			'icon'  => 'fa-youtube',
		),
		array(
			'name'  => 'vimeo',
			'label' => esc_html__( 'Vimeo', 'public-opinion-lite' ),
			'icon'  => 'fa-vimeo',
		),
		array(
			'name'  => 'gplus',
			'label' => esc_html__( 'Google Plus', 'public-opinion-lite' ),
			'icon'  => 'fa-google-plus',
		),
		array(
			'name'  => 'linkedin',
			'label' => esc_html__( 'LinkedIn', 'public-opinion-lite' ),
			'icon'  => 'fa-linkedin',
		),
		array(
			'name'  => 'tumblr',
			'label' => esc_html__( 'Tumblr', 'public-opinion-lite' ),
			'icon'  => 'fa-tumblr',
		),
		array(
			'name'  => 'flickr',
			'label' => esc_html__( 'Flickr', 'public-opinion-lite' ),
			'icon'  => 'fa-flickr',
		),
		array(
			'name'  => 'dribbble',
			'label' => esc_html__( 'Dribbble', 'public-opinion-lite' ),
			'icon'  => 'fa-dribbble',
		),
		array(
			'name'  => 'wordpress',
			'label' => esc_html__( 'WordPress', 'public-opinion-lite' ),
			'icon'  => 'fa-wordpress',
		),
		array(
			'name'  => '500px',
			'label' => esc_html__( '500px', 'public-opinion-lite' ),
			'icon'  => 'fa-500px',
		),
		array(
			'name'  => 'soundcloud',
			'label' => esc_html__( 'Soundcloud', 'public-opinion-lite' ),
			'icon'  => 'fa-soundcloud',
		),
		array(
			'name'  => 'spotify',
			'label' => esc_html__( 'Spotify', 'public-opinion-lite' ),
			'icon'  => 'fa-spotify',
		),
		array(
			'name'  => 'vine',
			'label' => esc_html__( 'Vine', 'public-opinion-lite' ),
			'icon'  => 'fa-vine',
		),
		array(
			'name'  => 'tripadvisor',
			'label' => esc_html__( 'Trip Advisor', 'public-opinion-lite' ),
			'icon'  => 'fa-tripadvisor',
		),
	) );
}


add_filter( 'tiny_mce_before_init', 'public_opinion_lite_base_insert_wp_editor_formats' );
function public_opinion_lite_base_insert_wp_editor_formats( $init_array ) {
	$style_formats = array(
		array(
			'title'   => esc_html__( 'Intro text (big text)', 'public-opinion-lite' ),
			'block'   => 'div',
			'classes' => 'entry-content-intro',
			'wrapper' => true,
		),
		array(
			'title'   => esc_html__( '2 Column Text', 'public-opinion-lite' ),
			'block'   => 'div',
			'classes' => 'entry-content-column-split',
			'wrapper' => true,
		),
	);
	$init_array['style_formats'] = wp_json_encode( $style_formats );

	return $init_array;
}

add_filter( 'mce_buttons_2', 'public_opinion_lite_base_mce_buttons_2' );
function public_opinion_lite_base_mce_buttons_2( $buttons ) {
	array_unshift( $buttons, 'styleselect' );

	return $buttons;
}


if ( ! function_exists( 'public_opinion_lite_color_luminance' ) ) :
	/**
	 * Lightens/darkens a given colour (hex format), returning the altered colour in hex format.
	 *
	 * @see https://gist.github.com/stephenharris/5532899
	 *
	 * @param string $color Hexadecimal color value. May be 3 or 6 digits, with an optional leading # sign.
	 * @param float $percent Decimal (0.2 = lighten by 20%, -0.4 = darken by 40%)
	 *
	 * @return string Lightened/Darkened colour as hexadecimal (with hash)
	 */
	function public_opinion_lite_color_luminance( $color, $percent ) {
		// Remove # if provided
		if ( '#' === $color[0] ) {
			$color = substr( $color, 1 );
		}

		// Validate hex string.
		$hex     = preg_replace( '/[^0-9a-f]/i', '', $color );
		$new_hex = '#';

		$percent = floatval( $percent );

		if ( strlen( $hex ) < 6 ) {
			$hex = $hex[0] + $hex[0] + $hex[1] + $hex[1] + $hex[2] + $hex[2];
		}

		// Convert to decimal and change luminosity.
		for ( $i = 0; $i < 3; $i ++ ) {
			$dec = hexdec( substr( $hex, $i * 2, 2 ) );
			$dec = min( max( 0, $dec + $dec * $percent ), 255 );
			$new_hex .= str_pad( dechex( $dec ), 2, 0, STR_PAD_LEFT );
		}

		return $new_hex;
	}
endif;

if ( ! function_exists( 'public_opinion_lite_hex2rgba' ) ) :
	/**
	 * Converts hexadecimal color value to rgb(a) format.
	 *
	 * @param string $color Hexadecimal color value. May be 3 or 6 digits, with an optional leading # sign.
	 * @param float|bool $opacity Opacity level 0-1 (decimal) or false to disable.
	 *
	 * @return string
	 */
	function public_opinion_lite_hex2rgba( $color, $opacity = false ) {

		$default = 'rgb(0,0,0)';

		// Return default if no color provided
		if ( empty( $color ) ) {
			return $default;
		}

		// Remove # if provided
		if ( '#' === $color[0] ) {
			$color = substr( $color, 1 );
		}

		// Check if color has 6 or 3 characters and get values
		if ( strlen( $color ) === 6 ) {
			$hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
		} elseif ( strlen( $color ) === 3 ) {
			$hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
		} else {
			return $default;
		}

		$rgb = array_map( 'hexdec', $hex );

		if ( false === $opacity ) {
			$opacity = abs( floatval( $opacity ) );
			if ( $opacity > 1 ) {
				$opacity = 1.0;
			}
			$output = 'rgba(' . implode( ',', $rgb ) . ',' . $opacity . ')';
		} else {
			$output = 'rgb(' . implode( ',', $rgb ) . ')';
		}

		return $output;
	}
endif;


/**
 * Return default args for add_theme_support( 'public-opinion-lite-hero' )
 *
 * Used when declaring support for theme hero section, so that unchanged args can be omitted. E.g.:
 *
 *  	add_theme_support( 'public-opinion-lite-hero', apply_filters( 'public_opinion_lite_theme_support_hero_defaults', wp_parse_args( array(
 *  		'required' => true,
 *  	), public_opinion_lite_base_theme_support_hero_defaults() ) ) );
 *
 * @return array
 */
function public_opinion_lite_base_theme_support_hero_defaults() {
	return apply_filters( 'public_opinion_lite_base_theme_support_hero_defaults', array(
		'required'               => false, // When true, there will be no option to hide the hero section.
		'show-default'           => false, // The default state of the 'hero_show' option.
		'show-if-text-empty'     => false, // Show hero when title and subtitle are empty. If 'required' = true this is ignored (and hero is always shown).
		'image-size'             => 'public_opinion_lite_hero', // The default image size for the background image.
		'front-page-template'    => 'templates/front-page.php', // The front page template slug. Set to false if theme doesn't have a front page template.
		'front-page-classes'     => '', // Extra hero classes for the front page.
		'front-page-image-size'  => false, // The image size for the front page, if different. False means same as 'image-size'.
		'text-align'             => 'left', // The default text-align for the hero text. One of: 'left', 'center', 'right'.
	) );
}

function public_opinion_lite_the_hero_classes( $echo = true ) {
	$classes = array( 'page-hero' );

	$hero_support = get_theme_support( 'public-opinion-lite-hero' );
	$hero_support = $hero_support[0];
	if ( $hero_support['front-page-template'] && is_page_template( $hero_support['front-page-template'] ) ) {
		$classes[] = $hero_support['front-page-classes'];
	}

	$classes = apply_filters( 'public_opinion_lite_hero_classes', $classes );
	$classes = array_filter( $classes );
	if ( $echo ) {
		echo esc_attr( implode( ' ', $classes ) );
	} else {
		return $classes;
	}
}

function public_opinion_lite_get_hero_data( $post_id = false ) {
	if ( is_singular() && false === $post_id ) {
		$post_id = get_the_ID();
	}

	if ( ! current_theme_supports( 'public-opinion-lite-hero' ) ) {
		return array(
			'show'            => 0,
			'page_title_hide' => 0,
		);
	}

	$support = get_theme_support( 'public-opinion-lite-hero' );
	$support = $support[0];

	$video_url = get_theme_mod( 'hero_video_url' );

	$title    = '';
	$subtitle = '';

	$image_size = $support['image-size'];
	if ( $support['front-page-image-size'] && is_page_template( $support['front-page-template'] ) ) {
		$image_size = $support['front-page-image-size'];
	}

	if ( is_home() ) {
		$title = get_theme_mod( 'title_blog', __( 'From the blog', 'public-opinion-lite' ) );
	} elseif ( is_search() ) {
		$title = get_theme_mod( 'title_search', __( 'Search results', 'public-opinion-lite' ) );

		global $wp_query;
		$found = intval( $wp_query->found_posts );
		/* translators: %d is the number of search results. */
		$subtitle = esc_html( sprintf( _n( '%d result found.', '%d results found.', $found, 'public-opinion-lite' ), $found ) );

	} elseif ( is_404() ) {
		$title = get_theme_mod( 'title_404', __( 'Page not found', 'public-opinion-lite' ) );
	} elseif ( is_category() || is_tag() || is_tax() ) {
		$title    = single_term_title( '', false );
		$subtitle = term_description();
	} elseif ( is_post_type_archive() ) {
		$title = post_type_archive_title( '', false );
	} elseif ( is_archive() ) {
		$title = get_the_archive_title();
	}

	$generic_data = array(
		'show'             => get_theme_mod( 'hero_show', $support['show-default'] ),
		'title'            => $title,
		'subtitle'         => $subtitle,
		'text_align'       => $support['text-align'],
		'page_title_hide'  => 0,
		'bg_color'         => get_theme_mod( 'hero_bg_color' ),
		'text_color'       => get_theme_mod( 'hero_text_color' ),
		'overlay_color'    => get_theme_mod( 'hero_overlay_color' ),
		'image_id'         => '',
		'image'            => get_theme_mod( 'hero_image' ),
		'image_repeat'     => get_theme_mod( 'hero_image_repeat', 'no-repeat' ),
		'image_position_x' => get_theme_mod( 'hero_image_position_x', 'center' ),
		'image_position_y' => get_theme_mod( 'hero_image_position_y', 'center' ),
		'image_attachment' => get_theme_mod( 'hero_image_attachment', 'scroll' ),
		'image_cover'      => get_theme_mod( 'hero_image_cover', 1 ),
		'video_url'        => $video_url,
		'video_info'       => public_opinion_lite_get_video_url_info( $video_url ),
	);

	$data = $generic_data;

	$single_data = array();

	if ( class_exists( 'WooCommerce' ) && ( is_shop() || is_product_taxonomy() || is_product() ) ) {
		// The conditionals can only be used AFTER the 'posts_selection' action (i.e. in 'wp'), so calling this function earlier,
		// e.g. on 'init' will not work properly. In that case, provide the shop's page ID explicitly when calling.
		// Example usage can be found on the Spencer theme.
		$shop_page = wc_get_page_id( 'shop' );
		if ( $shop_page > 0 ) {
			$post_id = $shop_page;
		}
	}

	if ( is_singular() || false !== $post_id ) {
		$image_id  = get_post_meta( $post_id, 'hero_image_id', true );
		$video_url = get_post_meta( $post_id, 'hero_video_url', true );

		$single_data = array(
			'show'             => get_post_meta( $post_id, 'hero_show', true ),
			'title'            => get_the_title( $post_id ), // May be custom title from hooked public_opinion_lite_base_replace_the_title()
			'subtitle'         => get_post_meta( $post_id, 'subtitle', true ),
			'text_align'       => get_post_meta( $post_id, 'hero_text_align', true ),
			'page_title_hide'  => get_post_meta( $post_id, 'page_title_hide', true ),
			'bg_color'         => get_post_meta( $post_id, 'hero_bg_color', true ),
			'text_color'       => get_post_meta( $post_id, 'hero_text_color', true ),
			'overlay_color'    => get_post_meta( $post_id, 'hero_overlay_color', true ),
			'video_url'        => $video_url,
			'video_info'       => public_opinion_lite_get_video_url_info( $video_url ),
			'image_id'         => $image_id,
			'image'            => $image_id ? wp_get_attachment_image_url( $image_id, $image_size ) : '',
			'image_repeat'     => get_post_meta( $post_id, 'hero_image_repeat', true ),
			'image_position_x' => get_post_meta( $post_id, 'hero_image_position_x', true ),
			'image_position_y' => get_post_meta( $post_id, 'hero_image_position_y', true ),
			'image_attachment' => get_post_meta( $post_id, 'hero_image_attachment', true ),
			'image_cover'      => get_post_meta( $post_id, 'hero_image_cover', true ),
		);

		if ( ! $single_data['page_title_hide'] ) {
			if ( is_singular( 'post' ) || ( false === $post_id && 'post' === get_post_type( $post_id ) ) ) {
				$single_data['title'] = get_theme_mod( 'title_blog', __( 'From the blog', 'public-opinion-lite' ) );
			}
		}

		$data = $single_data;

		$data['title']    = trim( $data['title'] );
		$data['subtitle'] = trim( $data['subtitle'] );

		// Hero is required, so lets inherit some values for best appearance.
		if ( $support['required'] ) {
			if ( empty( $data['text_align'] ) ) {
				$data['text_align'] = $generic_data['text_align'];
			}
			if ( empty( $data['bg_color'] ) ) {
				$data['bg_color'] = $generic_data['bg_color'];
			}
			if ( empty( $data['text_color'] ) ) {
				$data['text_color'] = $generic_data['text_color'];
			}
			if ( empty( $data['overlay_color'] ) ) {
				$data['overlay_color'] = $generic_data['overlay_color'];
			}
			if ( empty( $data['video_url'] ) ) {
				$data['video_url']  = $generic_data['video_url'];
				$data['video_info'] = $generic_data['video_info'];
			}
			if ( empty( $data['image_id'] ) ) {
				$data['image_id']         = $generic_data['image_id'];
				$data['image']            = $generic_data['image'];
				$data['image_repeat']     = $generic_data['image_repeat'];
				$data['image_position_x'] = $generic_data['image_position_x'];
				$data['image_position_y'] = $generic_data['image_position_y'];
				$data['image_attachment'] = $generic_data['image_attachment'];
				$data['image_cover']      = $generic_data['image_cover'];
			}
		}

	}

	if ( class_exists( 'WooCommerce' ) ) {
		if ( is_product() ) {
			$data['title']    = get_the_title( $shop_page ); // May be custom title from hooked public_opinion_lite_base_replace_the_title()
			$data['subtitle'] = get_post_meta( $shop_page, 'subtitle', true );
		} elseif ( is_product_taxonomy() ) {
			$data['title']    = single_term_title( '', false );
			$data['subtitle'] = term_description();
		}
	}

	// Disable hero if no text exists.
	if ( false === $support['show-if-text-empty'] && empty( $data['title'] ) && empty( $data['subtitle'] ) ) {
		$data['show'] = 0;
	}

	// Enable hero if required, ignoring previous limitations ( e.g. false === $support['show-if-text-empty'] ).
	if ( $support['required'] ) {
		$data['show'] = 1;
	}

	return apply_filters( 'public_opinion_lite_hero_data', $data, $post_id, $generic_data, $single_data );
}


function public_opinion_lite_get_video_url_info( $url ) {
	$is_vimeo   = preg_match( '#(?:https?://)?(?:www\.)?vimeo\.com/([A-Za-z0-9\-_]+)#', $url, $vimeo_id );
	$is_youtube = preg_match( '~
		# Match non-linked youtube URL in the wild. (Rev:20111012)
		https?://         # Required scheme. Either http or https.
		(?:[0-9A-Z-]+\.)? # Optional subdomain.
		(?:               # Group host alternatives.
		  youtu\.be/      # Either youtu.be,
		| youtube\.com    # or youtube.com followed by
		  \S*             # Allow anything up to VIDEO_ID,
		  [^\w\-\s]       # but char before ID is non-ID char.
		)                 # End host alternatives.
		([\w\-]{11})      # $1: VIDEO_ID is exactly 11 chars.
		(?=[^\w\-]|$)     # Assert next char is non-ID or EOS.
		(?!               # Assert URL is not pre-linked.
		  [?=&+%\w]*      # Allow URL (query) remainder.
		  (?:             # Group pre-linked alternatives.
			[\'"][^<>]*>  # Either inside a start tag,
		  | </a>          # or inside <a> element text contents.
		  )               # End recognized pre-linked alts.
		)                 # End negative lookahead assertion.
		[?=&+%\w-]*        # Consume any URL (query) remainder.
		~ix',
	$url, $youtube_id );

	$info = array(
		'supported' => false,
		'provider'  => '',
		'video_id'  => '',
	);

	if ( $is_youtube ) {
		$info['supported'] = true;
		$info['provider']  = 'youtube';
		$info['video_id']  = $youtube_id[1];
	} elseif ( $is_vimeo ) {
		$info['supported'] = true;
		$info['provider']  = 'vimeo';
		$info['video_id']  = $vimeo_id[1];
	}

	return $info;
}


/**
 * Returns a set of related posts, or the arguments needed for such a query.
 *
 * @uses wp_parse_args()
 * @uses get_post_type()
 * @uses get_post()
 * @uses get_object_taxonomies()
 * @uses get_the_terms()
 * @uses wp_list_pluck()
 *
 * @param int $post_id A post ID to get related posts for.
 * @param int $related_count The number of related posts to return.
 * @param array $args Array of arguments to change the default behavior.
 * @return object|array A WP_Query object with the results, or an array with the query arguments.
 */
function public_opinion_lite_get_related_posts( $post_id, $related_count, $args = array() ) {
	$args = wp_parse_args( (array) $args, array(
		'orderby' => 'rand',
		'return'  => 'query', // Valid values are: 'query' (WP_Query object), 'array' (the arguments array)
	) );

	$post_type = get_post_type( $post_id );
	$post      = get_post( $post_id );

	$tax_query  = array();
	$taxonomies = get_object_taxonomies( $post, 'names' );

	foreach ( $taxonomies as $taxonomy ) {
		$terms = get_the_terms( $post_id, $taxonomy );
		if ( is_array( $terms ) && count( $terms ) > 0 ) {
			$term_list = wp_list_pluck( $terms, 'slug' );
			$term_list = array_values( $term_list );
			if ( ! empty( $term_list ) ) {
				$tax_query['tax_query'][] = array(
					'taxonomy' => $taxonomy,
					'field'    => 'slug',
					'terms'    => $term_list,
				);
			}
		}
	}

	if ( count( $taxonomies ) > 1 ) {
		$tax_query['tax_query']['relation'] = 'OR';
	}

	$query_args = array(
		'post_type'      => $post_type,
		'posts_per_page' => $related_count,
		'post_status'    => 'publish',
		'post__not_in'   => array( $post_id ),
		'orderby'        => $args['orderby'],
	);

	if ( 'query' === $args['return'] ) {
		return new WP_Query( array_merge( $query_args, $tax_query ) );
	} else {
		return array_merge( $query_args, $tax_query );
	}
}


/**
 * Returns the appropriate page(d) query variable to use in custom loops (needed for pagination).
 *
 * @uses get_query_var()
 *
 * @param int $default_return The default page number to return, if no query vars are set.
 * @return int The appropriate paged value if found, else 0.
 */
function public_opinion_lite_get_page_var( $default_return = 0 ) {
	$paged = get_query_var( 'paged', false );
	$page  = get_query_var( 'page', false );

	if ( false === $paged && false === $page ) {
		return $default_return;
	}

	return max( $paged, $page );
}


/**
 * Retrieve or display list of posts as a dropdown (select list).
 *
 * @since 2.1.0
 *
 * @param array|string $args Optional. Override default arguments.
 * @param string $name Optional. Name of the select box.
 * @return string HTML content, if not displaying.
 */
function public_opinion_lite_dropdown_posts( $args = '', $name = 'post_id' ) {
	$defaults = array(
		'depth'                 => 0,
		'post_parent'           => 0,
		'selected'              => 0,
		'echo'                  => 1,
		//'name'                  => 'page_id', // With this line, get_posts() doesn't work properly.
		'id'                    => '',
		'class'                 => '',
		'show_option_none'      => '',
		'show_option_no_change' => '',
		'option_none_value'     => '',
		'post_type'             => 'post',
		'post_status'           => 'publish',
		'suppress_filters'      => false,
		'numberposts'           => -1,
		'select_even_if_empty'  => false, // If no posts are found, an empty <select> will be returned/echoed.
	);

	$r = wp_parse_args( $args, $defaults );
	extract( $r, EXTR_SKIP );

	$hierarchical_post_types = get_post_types( array( 'hierarchical' => true ) );
	if ( in_array( $r['post_type'], $hierarchical_post_types ) ) {
		$pages = get_pages($r);
	} else {
		$pages = get_posts($r);
	}

	$output = '';
	// Back-compat with old system where both id and name were based on $name argument
	if ( empty($id) )
		$id = $name;

	if ( ! empty($pages) || $select_even_if_empty == true ) {
		$output = "<select name='" . esc_attr( $name ) . "' id='" . esc_attr( $id ) . "' class='" . esc_attr( $class ) . "'>\n";
		if ( $show_option_no_change ) {
			$output .= "\t<option value=\"-1\">$show_option_no_change</option>";
		}
		if ( $show_option_none ) {
			$output .= "\t<option value=\"" . esc_attr( $option_none_value ) . "\">$show_option_none</option>\n";
		}
		if ( ! empty($pages) ) {
			$output .= walk_page_dropdown_tree($pages, $depth, $r);
		}
		$output .= "</select>\n";
	}

	$output = apply_filters( 'public_opinion_lite_dropdown_posts', $output, $name, $r );

	if ( $echo )
		echo $output;

	return $output;
}


add_action( 'admin_init', 'public_opinion_lite_admin_setup_hide_single_featured' );
function public_opinion_lite_admin_setup_hide_single_featured() {
	if ( current_theme_supports( 'public-opinion-lite-hide-single-featured' ) ) {
		$hide_featured_support = get_theme_support( 'public-opinion-lite-hide-single-featured' );
		$hide_featured_support = $hide_featured_support[0];

		foreach ( $hide_featured_support as $supported_post_type ) {
			add_meta_box( 'public-opinion-lite-single-featured-visibility', esc_html__( 'Featured Image Visibility', 'public-opinion-lite' ), 'public_opinion_lite_single_featured_visibility_metabox', $supported_post_type, 'side', 'default' );
		}
	}

	add_action( 'save_post', 'public_opinion_lite_hide_single_featured_save_post' );
}

add_action( 'init', 'public_opinion_lite_setup_hide_single_featured' );
function public_opinion_lite_setup_hide_single_featured() {
	if ( current_theme_supports( 'public-opinion-lite-hide-single-featured' ) ) {
		add_filter( 'get_post_metadata', 'public_opinion_lite_hide_single_featured_get_post_metadata', 10, 4 );
	}
}

function public_opinion_lite_single_featured_visibility_metabox( $object, $box ) {
	$fieldname = 'public_opinion_lite_hide_single_featured';
	$checked   = get_post_meta( $object->ID, $fieldname, true );

	?>
		<input type="checkbox" id="<?php echo esc_attr( $fieldname ); ?>" class="check" name="<?php echo esc_attr( $fieldname ); ?>" value="1" <?php checked( $checked, 1 ); ?> />
		<label for="<?php echo esc_attr( $fieldname ); ?>"><?php esc_html_e( "Hide when viewing this post's page", 'public-opinion-lite' ); ?></label>
	<?php
	wp_nonce_field( 'public_opinion_lite_hide_single_featured_nonce', '_public_opinion_lite_hide_single_featured_meta_box_nonce' );
}

function public_opinion_lite_hide_single_featured_get_post_metadata( $value, $post_id, $meta_key, $single ) {
	$hide_featured_support = get_theme_support( 'public-opinion-lite-hide-single-featured' );
	$hide_featured_support = $hide_featured_support[0];

	if ( ! in_array( get_post_type( $post_id ), $hide_featured_support, true ) ) {
		return $value;
	}

	if ( '_thumbnail_id' === $meta_key && ( is_single( $post_id ) || is_page( $post_id ) ) && get_post_meta( $post_id, 'public_opinion_lite_hide_single_featured', true ) ) {
		return false;
	}

	return $value;
}

function public_opinion_lite_hide_single_featured_save_post( $post_id ) {
	$hide_featured_support = get_theme_support( 'public-opinion-lite-hide-single-featured' );
	$hide_featured_support = $hide_featured_support[0];

	if ( ! in_array( get_post_type( $post_id ), $hide_featured_support, true ) ) {
		return;
	}

	if ( isset( $_POST['_public_opinion_lite_hide_single_featured_meta_box_nonce'] ) && wp_verify_nonce( $_POST['_public_opinion_lite_hide_single_featured_meta_box_nonce'], 'public_opinion_lite_hide_single_featured_nonce' ) ) { // Input vars okay.
		update_post_meta( $post_id, 'public_opinion_lite_hide_single_featured', isset( $_POST['public_opinion_lite_hide_single_featured'] ) ); // Input var okay.
	}
}



function public_opinion_lite_archive_layout_choices() {
	return apply_filters( 'public_opinion_lite_archive_layout_choices', array(
		1 => sprintf( _n( '%d column', '%d columns', 1, 'public-opinion-lite' ), number_format_i18n( 1 ) ),
		2 => sprintf( _n( '%d column', '%d columns', 2, 'public-opinion-lite' ), number_format_i18n( 2 ) ),
		3 => sprintf( _n( '%d column', '%d columns', 3, 'public-opinion-lite' ), number_format_i18n( 3 ) ),
	) );
}

function public_opinion_lite_archive_layout_default() {
	return apply_filters( 'public_opinion_lite_archive_layout_default', 1 );
}

function public_opinion_lite_sanitize_archive_layout( $value ) {
	$choices = public_opinion_lite_archive_layout_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return public_opinion_lite_archive_layout_default();
}

function public_opinion_lite_header_layout_choices() {
	$choices = array(
		'right' => esc_html__( 'Left logo - Right menu', 'public-opinion-lite' ),
		'split' => esc_html__( 'Centered logo - Split menu', 'public-opinion-lite' ),
		'full'  => esc_html__( 'Top logo - Bottom menu', 'public-opinion-lite' ),
	);

	if ( ! apply_filters( 'public_opinion_lite_base_support_menu_2', true ) ) {
		unset( $choices['split'] );
	}

	return apply_filters( 'public_opinion_lite_header_layout_choices', $choices );
}

function public_opinion_lite_header_layout_default() {
	return apply_filters( 'public_opinion_lite_header_layout_default', 'right' );
}

function public_opinion_lite_sanitize_header_layout( $value ) {
	$choices = public_opinion_lite_header_layout_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return public_opinion_lite_header_layout_default();
}

function public_opinion_lite_header_logo_alignment_choices() {
	return apply_filters( 'public_opinion_lite_header_logo_alignment_choices', array(
		'left'   => esc_html__( 'Left', 'public-opinion-lite' ),
		'center' => esc_html__( 'Center', 'public-opinion-lite' ),
	) );
}

function public_opinion_lite_header_logo_alignment_default() {
	return apply_filters( 'public_opinion_lite_header_logo_alignment_default', 'center' );
}

function public_opinion_lite_sanitize_header_logo_alignment( $value ) {
	$choices = public_opinion_lite_header_logo_alignment_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return public_opinion_lite_header_logo_alignment_default();
}


function public_opinion_lite_footer_layout_choices() {
	return apply_filters( 'public_opinion_lite_footer_layout_choices', array(
		'4-col' => esc_html__( '4 Columns', 'public-opinion-lite' ),
		'3-col' => esc_html__( '3 Columns', 'public-opinion-lite' ),
		'2-col' => esc_html__( '2 Columns', 'public-opinion-lite' ),
		'1-col' => esc_html__( '1 Column', 'public-opinion-lite' ),
		'1-3'   => esc_html__( '1/4 - 3/4 Columns', 'public-opinion-lite' ),
		'3-1'   => esc_html__( '3/4 - 1/4 Columns', 'public-opinion-lite' ),
		'1-1-2' => esc_html__( '1/4 - 1/4 - 1/2 Columns', 'public-opinion-lite' ),
		'2-1-1' => esc_html__( '1/2 - 1/4 - 1/4 Columns', 'public-opinion-lite' ),
	) );
}

function public_opinion_lite_footer_layout_default() {
	return apply_filters( 'public_opinion_lite_footer_layout_default', '4-col' );
}

function public_opinion_lite_sanitize_footer_layout( $value ) {
	$choices = public_opinion_lite_footer_layout_choices();
	if ( array_key_exists( $value, $choices ) ) {
		return $value;
	}

	return public_opinion_lite_footer_layout_default();
}

/**
 * Sanitizes the pagination method option.
 *
 * @param string $option Value to sanitize. Either 'numbers' or 'text'.
 * @return string
 */
function public_opinion_lite_sanitize_pagination_method( $option ) {
	if ( in_array( $option, array( 'numbers', 'text' ), true ) ) {
		return $option;
	}

	return public_opinion_lite_pagination_method_default();
}

function public_opinion_lite_pagination_method_default() {
	return apply_filters( 'public_opinion_lite_pagination_method_default', 'numbers' );
}

function public_opinion_lite_footer_widget_area_classes( $layout ) {
	switch ( $layout ) {
		case '3-col':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-4 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-4 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-4 col-12',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '2-col':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '1-col':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-12',
				),
				'footer-2' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '1-3':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-9 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '3-1':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-9 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => false,
					'class'  => '',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '1-1-2':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-6 col-12',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '2-1-1':
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-4' => array(
					'active' => false,
					'class'  => '',
				),
			);
			break;
		case '4-col':
		default:
			$classes = array(
				'footer-1' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-2' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-3' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
				'footer-4' => array(
					'active' => true,
					'class'  => 'col-lg-3 col-md-6 col-12',
				),
			);
	} // End switch().

	return apply_filters( 'public_opinion_lite_footer_widget_area_classes', $classes, $layout );
}


function public_opinion_lite_is_content_empty( $str ) {
	return trim( str_replace( '&nbsp;', '', strip_tags( $str ) ) ) == '';
}


/**
 * Returns the caption of an image, to be used in a lightbox.
 *
 * @uses get_post_thumbnail_id()
 * @uses wp_prepare_attachment_for_js()
 *
 * @param int|false $image_id The image's attachment ID.
 *
 * @return string
 */
function public_opinion_lite_get_image_lightbox_caption( $image_id = false ) {
	if ( false === $image_id ) {
		$image_id = get_post_thumbnail_id();
	}

	$lightbox_caption = '';

	$attachment = wp_prepare_attachment_for_js( $image_id );

	if ( is_array( $attachment ) ) {
		$field = apply_filters( 'public_opinion_lite_image_lightbox_caption_field', 'caption', $image_id, $attachment );

		if ( array_key_exists( $field, $attachment ) ) {
			$lightbox_caption = $attachment[ $field ];
		}
	}

	return $lightbox_caption;
}


/**
 * Standard template tags.
 */
require get_theme_file_path( '/base/template-tags.php' );

/**
 * Standard sanitization functions.
 */
require get_theme_file_path( '/base/sanitization.php' );

/**
 * Standard customizer controls.
 */
require get_theme_file_path( '/base/customizer.php' );

/**
 * Standard customizer partial callbacks.
 */
require get_theme_file_path( '/base/customizer-partial-callbacks.php' );

/**
 * Standard customizer styles.
 */
require get_theme_file_path( '/base/customizer-styles.php' );

/**
 * Standard hooks.
 */
require get_theme_file_path( '/base/default-hooks.php' );

